<?php
session_start();
include 'includes/db.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$error = "";
$success = "";

// Handle PIN submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $pin = trim($_POST['pin']);
    $confirm_pin = trim($_POST['confirm_pin']);

    // Validation
    if (empty($pin) || empty($confirm_pin)) {
        $error = "Please enter and confirm your PIN.";
    } elseif ($pin !== $confirm_pin) {
        $error = "PINs do not match.";
    } elseif (!preg_match('/^\d{4}$/', $pin)) {
        $error = "PIN must be exactly 4 digits.";
    } else {
        // Hash the PIN
        $hashed_pin = password_hash($pin, PASSWORD_DEFAULT);

        // Update PIN in database
        $sql = "UPDATE users SET pin = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("si", $hashed_pin, $user_id);

        if ($stmt->execute()) {
            $success = "Your PIN has been set/updated successfully.";
        } else {
            $error = "Failed to update PIN. Please try again.";
        }
    }
}

// Fetch user info (optional for display)
$sql = "SELECT name FROM users WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MYVTU - Set PIN</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        body {
            font-family: Arial, sans-serif;
            background-color: #f4f6f8;
            display: flex;
            flex-direction: column;
            min-height: 100vh;
            align-items: center;
            justify-content: center;
        }
        .pin-container {
            background: #fff;
            padding: 40px 30px;
            border-radius: 10px;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            text-align: center;
            width: 300px;
        }
        .pin-container h2 {
            color: #203a43;
            margin-bottom: 20px;
        }
        input[type="password"] {
            width: 100%;
            padding: 12px;
            margin: 10px 0;
            border: 1px solid #ccc;
            border-radius: 6px;
        }
        input[type="submit"] {
            width: 100%;
            padding: 12px;
            background-color: #203a43;
            color: #f4f6f8;
            border: none;
            border-radius: 6px;
            font-weight: bold;
            cursor: pointer;
        }
        input[type="submit"]:hover {
            background-color: #ff6600;
        }
        .error {
            color: red;
            margin-bottom: 15px;
        }
        .success {
            color: green;
            margin-bottom: 15px;
        }
        a.back-link {
            display: block;
            margin-top: 15px;
            text-decoration: none;
            color: #203a43;
        }
        a.back-link:hover {
            color: #ff6600;
        }
    </style>
</head>
<body>

<div class="pin-container">
    <h2>Hello, <?php echo htmlspecialchars($user['name']); ?>!</h2>
    <h3>Set or Update Your PIN</h3>

    <?php if ($error != ""): ?>
        <div class="error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <?php if ($success != ""): ?>
        <div class="success"><?php echo $success; ?></div>
    <?php endif; ?>

    <form method="POST" action="">
        <input type="password" name="pin" placeholder="Enter 4-digit PIN" required>
        <input type="password" name="confirm_pin" placeholder="Confirm PIN" required>
        <input type="submit" value="Set PIN">
    </form>

    <a class="back-link" href="dashboard.php">Back to Dashboard</a>
</div>

</body>
</html>