<?php
require_once "config/db.php";

/* ======================================================
   USER FUNCTIONS
====================================================== */

// Get a user by ID
function getUserById($user_id){
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM users WHERE id=?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    return $stmt->get_result()->fetch_assoc();
}

// Get a user by email
function getUserByEmail($email){
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM users WHERE email=?");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    return $stmt->get_result()->fetch_assoc();
}

// Get all users
function getAllUsers(){
    global $conn;
    return $conn->query("SELECT * FROM users ORDER BY id DESC");
}

// Update user wallet
function updateWallet($user_id, $amount){
    global $conn;
    $stmt = $conn->prepare("UPDATE users SET balance=? WHERE id=?");
    $stmt->bind_param("di", $amount, $user_id);
    $stmt->execute();
}

// Update referral bonus
function updateReferralBonus($user_id, $bonus){
    global $conn;
    $stmt = $conn->prepare("UPDATE users SET referral_bonus=? WHERE id=?");
    $stmt->bind_param("di", $bonus, $user_id);
    $stmt->execute();
}

/* ======================================================
   TRANSACTION FUNCTIONS
====================================================== */

// Add a transaction
function addTransaction($user_id, $service, $description, $amount, $status){
    global $conn;
    $stmt = $conn->prepare("INSERT INTO transactions (user_id, service, description, amount, status, created_at) VALUES (?,?,?,?,?,NOW())");
    $stmt->bind_param("issds", $user_id, $service, $description, $amount, $status);
    $stmt->execute();
}

// Get all transactions
function getAllTransactions(){
    global $conn;
    return $conn->query("SELECT t.*, u.fullname, u.email FROM transactions t JOIN users u ON t.user_id=u.id ORDER BY t.created_at DESC");
}

// Get transactions by user
function getTransactionsByUser($user_id){
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM transactions WHERE user_id=? ORDER BY created_at DESC");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    return $stmt->get_result();
}

/* ======================================================
   SERVICE FUNCTIONS
====================================================== */

// Get all service pricing
function getAllServicePricing(){
    global $conn;
    return $conn->query("SELECT * FROM service_pricing ORDER BY service ASC");
}

// Get service price by service & provider
function getServicePrice($service, $provider){
    global $conn;
    $stmt = $conn->prepare("SELECT price FROM service_pricing WHERE service=? AND provider=? LIMIT 1");
    $stmt->bind_param("ss", $service, $provider);
    $stmt->execute();
    $res = $stmt->get_result()->fetch_assoc();
    return $res ? $res['price'] : 0;
}

// Update service price
function updateServicePrice($service, $provider, $price){
    global $conn;
    // Check if exists
    $stmt = $conn->prepare("SELECT id FROM service_pricing WHERE service=? AND provider=? LIMIT 1");
    $stmt->bind_param("ss", $service, $provider);
    $stmt->execute();
    $res = $stmt->get_result();
    if($res->num_rows > 0){
        $row = $res->fetch_assoc();
        $stmt = $conn->prepare("UPDATE service_pricing SET price=? WHERE id=?");
        $stmt->bind_param("di", $price, $row['id']);
        $stmt->execute();
    } else {
        $stmt = $conn->prepare("INSERT INTO service_pricing (service, provider, price) VALUES (?,?,?)");
        $stmt->bind_param("ssd", $service, $provider, $price);
        $stmt->execute();
    }
}

/* ======================================================
   WALLET FUNCTIONS
====================================================== */

// Get total wallet balance
function getTotalWallet(){
    global $conn;
    return $conn->query("SELECT SUM(balance) AS total FROM users")->fetch_assoc()['total'];
}

// Get total referral bonus
function getTotalReferralBonus(){
    global $conn;
    return $conn->query("SELECT SUM(referral_bonus) AS total FROM users")->fetch_assoc()['total'];
}

/* ======================================================
   REFERRAL FUNCTIONS
====================================================== */

// Get referral bonus for a user
function getReferralBonus($user_id){
    global $conn;
    $stmt = $conn->prepare("SELECT referral_bonus FROM users WHERE id=?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $res = $stmt->get_result()->fetch_assoc();
    return $res ? $res['referral_bonus'] : 0;
}