<?php
require_once "config/db.php";

/**
 * Fetch user details
 */
function getUser($user_id) {
    global $conn;
    $stmt = $conn->prepare("SELECT * FROM users WHERE id=?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    return $stmt->get_result()->fetch_assoc();
}

/**
 * Update user balance
 */
function updateBalance($user_id, $amount) {
    global $conn;
    $stmt = $conn->prepare("UPDATE users SET balance=? WHERE id=?");
    $stmt->bind_param("di", $amount, $user_id);
    return $stmt->execute();
}

/**
 * Add transaction
 */
function addTransaction($user_id, $service, $description, $amount, $status="SUCCESS") {
    global $conn;
    $stmt = $conn->prepare(
        "INSERT INTO transactions (user_id, service, description, amount, status) VALUES (?,?,?,?,?)"
    );
    $stmt->bind_param("issds", $user_id, $service, $description, $amount, $status);
    return $stmt->execute();
}

/**
 * Credit wallet
 */
function creditWallet($user_id, $amount, $desc="Wallet Credit") {
    $user = getUser($user_id);
    $new_balance = $user['balance'] + $amount;
    updateBalance($user_id, $new_balance);
    addTransaction($user_id, "FUND_WALLET", $desc, $amount, "SUCCESS");
    return $new_balance;
}

/**
 * Debit wallet
 */
function debitWallet($user_id, $amount, $service="SERVICE", $desc="Wallet Debit") {
    $user = getUser($user_id);
    if($user['balance'] < $amount) {
        return false; // insufficient funds
    }
    $new_balance = $user['balance'] - $amount;
    updateBalance($user_id, $new_balance);
    addTransaction($user_id, $service, $desc, $amount, "SUCCESS");
    return $new_balance;
}

/**
 * Generate random referral code
 */
function generateReferralCode($length=6) {
    return strtoupper(substr(bin2hex(random_bytes($length)), 0, $length));
}

/**
 * Calculate referral bonus
 */
function addReferralBonus($referrer_id, $bonus_amount) {
    global $conn;
    $stmt = $conn->prepare("UPDATE users SET referral_bonus = referral_bonus + ? WHERE id=?");
    $stmt->bind_param("di", $bonus_amount, $referrer_id);
    return $stmt->execute();
}

/**
 * Format currency
 */
function formatCurrency($amount) {
    return "₦" . number_format($amount, 2);
}
?>